#' ggplot_multi_lpi
#'
#' @param lpis - The list of lpis to plot
#' @param names - The names of the lpis in the list
#' @param ylims - The ylims of each plot
#' @param xlims - the xlims of each plot
#' @param title - the title of the plots (same for all)
#' @param col - the RColorBrewer Set to use. Default is "Set2"
#' @param facet - Whether or not to 'facet' the plot (or overlay)
#' @param yrbreaks - The spacing between x-axis tick marks
#' @param lpi_breaks - The spacing between y-axis tick marks
#'
#' @return Returns the calculated plot
#' @export
#'
ggplot_multi_lpi <- function(lpis, names=NULL,
                             ylims=c(0, 2), xlims=NULL,
                             title="", col=c("lightseagreen","coral"),
                             facet=FALSE, trans="identity",
                             yrbreaks = 5,
                             lpi_breaks = 0.2,
                             alpha = 0.5) {

  # Bit of a hack to avoid NOTE during R CMD check
  # Sets the variables used in ggplot2::aes to NULL
  years <- lpi <- group <- lwr <- upr <- NULL

  dfs <- data.frame(years=numeric(0), lpi=numeric(0), lwr=numeric(0), upr=numeric(0), group=character(0))

  if (is.null(names)) {
    names = LETTERS[1:length(lpis)]
  }
  for (i in 1:length(lpis)) {
    d <- cbind(lpis[[i]], group=names[i])
    df <- data.frame(years=as.numeric(as.character(rownames(d))), lpi=d$LPI_final, lwr=d$CI_low, upr=d$CI_high, group=d$group)
    dfs = rbind(dfs, df)
  }

  #print(dfs)
  g = ggplot2::ggplot(dfs, ggplot2::aes(x=years, y=lpi, group=group))+
    ggplot2::geom_hline(yintercept=1, alpha=1) +
    ggplot2::geom_line(ggplot2::aes(color=factor(group, levels=c(unique(group)[1],unique(group)[2]))), size=0.7) +
    ggplot2::geom_ribbon(ggplot2::aes(ymin=lwr,ymax=upr, fill=factor(group, levels=c(unique(group)[1],unique(group)[2]))),alpha=alpha,linetype=0) +
    ggplot2::coord_cartesian(ylim=ylims, xlim=xlims) + ggplot2::theme_bw() +
    ggplot2::theme(text = ggplot2::element_text(size=18),
                   axis.text.x = ggplot2::element_text(size=10, angle = 90, hjust = 1)) +
    ggplot2::ggtitle(title) +
    ggplot2::scale_fill_manual(values=col) +
    ggplot2::scale_color_manual(values=col) +
    ggplot2::ylab("Index") +
    ggplot2::xlab(element_blank()) +
    ggplot2::scale_y_continuous(trans=trans, breaks=seq(ylims[1], ylims[2], lpi_breaks)) +
    ggplot2::scale_x_continuous(breaks=seq(xlims[1], xlims[2], yrbreaks)) +
    ggplot2::theme(legend.position=c(0.87, 0.93), legend.background = element_blank(), legend.title = element_blank())


  if (facet) {
    g <- g + ggplot2::facet_grid( ~ factor(group, levels=c(unique(group)[1],unique(group)[2])))
  }
  print(g)
}


